<?php
/*
Plugin Name: Auto RSS to WP Posts
Description: Automatically import posts from multiple RSS feeds into specific WordPress categories without duplicates. Each feed can have a language assigned.
Version: 1.18
Author: Rasheed Hassan
*/

if (!defined('ABSPATH')) exit;

class AutoRSSImporter {
    private $option_name = 'auto_rss_import_feeds';
    private $log_option_name = 'auto_rss_import_logs';
    private $interval_option_name = 'auto_rss_import_interval';
    private $user_option_name = 'auto_rss_import_user';
    private $fetch_full_content_option = 'auto_rss_import_fetch_full';
    private $max_posts_option = 'auto_rss_import_max_items';
    private $item_limit = 10;

    public function __construct() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('auto_rss_import_hook', [$this, 'import_feeds']);

        add_filter('cron_schedules', [$this, 'custom_cron_schedules']);
        $this->schedule_cron();

        if (is_admin()) {
            add_action('wp_ajax_manual_import_feed', [$this, 'manual_import_feed']);
            add_action('admin_footer', [$this, 'inject_ajaxurl']);
        }
    }

    public function inject_ajaxurl() {
        echo '<script type="text/javascript">var ajaxurl = "' . admin_url('admin-ajax.php') . '";</script>';
    }

    public function add_admin_menu() {
        add_options_page('Auto RSS Importer', 'Auto RSS Importer', 'manage_options', 'auto-rss-importer', [$this, 'settings_page']);
    }

    public function register_settings() {
        register_setting('auto_rss_importer_group', $this->option_name);
        register_setting('auto_rss_importer_group', $this->interval_option_name);
        register_setting('auto_rss_importer_group', $this->user_option_name);
        register_setting('auto_rss_importer_group', $this->fetch_full_content_option);
        register_setting('auto_rss_importer_group', $this->max_posts_option);
    }

    public function custom_cron_schedules($schedules) {
        $schedules['every15min'] = ['interval' => 900, 'display' => __('Every 15 Minutes')];
        $schedules['every30min'] = ['interval' => 1800, 'display' => __('Every 30 Minutes')];
        $schedules['every1hour'] = ['interval' => 3600, 'display' => __('Every 1 Hour')];
        $schedules['every6hours'] = ['interval' => 21600, 'display' => __('Every 6 Hours')];
        $schedules['every12hours'] = ['interval' => 43200, 'display' => __('Every 12 Hours')];
        return $schedules;
    }

    private function schedule_cron() {
        $interval = get_option($this->interval_option_name, 'hourly');
        if (!wp_next_scheduled('auto_rss_import_hook')) {
            wp_schedule_event(time(), $interval, 'auto_rss_import_hook');
        }
    }

    public function manual_import_feed() {
        if (!current_user_can('manage_options')) wp_send_json_error('Unauthorized');

        $url = isset($_GET['url']) ? esc_url_raw($_GET['url']) : '';
        if (!$url) wp_send_json_error('Missing feed URL');

        $cat = isset($_GET['cat']) ? intval($_GET['cat']) : 1;
        $lang = isset($_GET['lang']) ? sanitize_text_field($_GET['lang']) : '';

        $this->import_feeds([['url' => $url, 'cat' => $cat, 'lang' => $lang]]);

        wp_send_json_success('Feed imported: ' . $url);
    }

    public function import_feeds($manual_urls = []) {
        $feeds = $manual_urls ?: get_option($this->option_name, []);
        $user_id = get_option($this->user_option_name);
        $fetch_full_content = get_option($this->fetch_full_content_option, true);
        $item_limit = get_option($this->max_posts_option, 10);

        if (!empty($feeds) && is_array($feeds)) {
            foreach ($feeds as $feed) {
                if (!is_array($feed) || !isset($feed['url'])) continue;

                $url = trim($feed['url']);
                $category_id = isset($feed['cat']) ? (int)$feed['cat'] : 1;
                $language = isset($feed['lang']) ? sanitize_text_field($feed['lang']) : '';

                if (empty($url)) continue;

                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0');
                $response = curl_exec($ch);
                curl_close($ch);

                if (!$response) continue;
                $xml = @simplexml_load_string($response, null, LIBXML_NOCDATA);
                if (!$xml) continue;

                $items = [];
                if (!empty($xml->channel->item)) {
                    $items = $xml->channel->item;
                } elseif (!empty($xml->item)) {
                    $items = $xml->item;
                }

                $count = 0;
                foreach ($items as $item) {
                    if ($count >= $item_limit) break;

                    $title = (string) $item->title;
                    $link  = (string) $item->link;
                    $guid  = (string) ($item->guid ?: $link);
                    $content = '';

                    if (isset($item->children('content', true)->encoded)) {
                        $content = (string)$item->children('content', true)->encoded;
                    } elseif (isset($item->description)) {
                        $content = (string)$item->description;
                    }

                    if ($fetch_full_content && $link) {
                        $html = wp_remote_get($link);
                        if (!is_wp_error($html) && !empty($html['body'])) {
                            $doc = new DOMDocument();
                            @$doc->loadHTML($html['body']);
                            $xpath = new DOMXPath($doc);

                            $nodes = $xpath->query('//article | //*[@class="post-content"] | //*[@class="entry-content"]');
                            if ($nodes->length > 0) {
                                $content = '';
                                foreach ($nodes as $node) {
                                    $content .= $doc->saveHTML($node);
                                }
                            }
                        }
                    }

                    $pubDate = !empty($item->pubDate) ? strtotime($item->pubDate) : current_time('timestamp');

                    if ($this->is_duplicate_by_meta($guid, $title)) continue;

                    $post_id = wp_insert_post([
                        'post_title'   => $title,
                        'post_content' => $content,
                        'post_status'  => 'publish',
                        'post_category'=> [$category_id],
                        'post_author'  => $user_id ?: get_current_user_id(),
                        'post_date'    => date('Y-m-d H:i:s', $pubDate),
                        'post_date_gmt'=> get_gmt_from_date(date('Y-m-d H:i:s', $pubDate)),
                    ]);

                    if ($post_id) {
                        add_post_meta($post_id, '_rss_item_link', esc_url_raw($guid));
                        if ($language) {
                            add_post_meta($post_id, '_rss_import_lang', $language);
                        }
                        $this->log("Imported: $title from $url");
                        $count++;
                    }
                }
            }
        }
    }

    private function is_duplicate_by_meta($guid, $title) {
        global $wpdb;

        $guid = esc_url_raw($guid);

        $post_id = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_rss_item_link' AND meta_value = %s LIMIT 1",
            $guid
        ));

        if (!empty($post_id)) return true;

        $recent_post = $wpdb->get_var($wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts} WHERE post_title = %s AND post_date > %s LIMIT 1",
            $title,
            date('Y-m-d H:i:s', strtotime('-30 days'))
        ));

        return !empty($recent_post);
    }

    private function log($message) {
        $logs = get_option($this->log_option_name, []);
        if (!is_array($logs)) $logs = [];
        $logs[] = current_time('mysql') . " - " . $message;
        if (count($logs) > 100) {
            $logs = array_slice($logs, -100);
        }
        update_option($this->log_option_name, $logs);
    }

    public function settings_page() {
        $feeds = get_option($this->option_name, []);
        ?>
        <div class="wrap">
            <h1>Auto RSS Importer Settings</h1>
            <form method="post" action="options.php">
                <?php settings_fields('auto_rss_importer_group'); ?>
                <?php do_settings_sections('auto_rss_importer_group'); ?>
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row">Fetch Full Article Content</th>
                        <td>
                            <input type="checkbox" name="<?= $this->fetch_full_content_option ?>" value="1" <?= checked(get_option($this->fetch_full_content_option, true), 1, false) ?> />
                            <label for="<?= $this->fetch_full_content_option ?>">Enable fetching full content from the original article link</label>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Import Interval</th>
                        <td>
                            <select name="<?= $this->interval_option_name ?>">
                                <option value="every15min" <?= selected(get_option($this->interval_option_name), 'every15min') ?>>Every 15 Minutes</option>
                                <option value="every30min" <?= selected(get_option($this->interval_option_name), 'every30min') ?>>Every 30 Minutes</option>
                                <option value="every1hour" <?= selected(get_option($this->interval_option_name), 'every1hour') ?>>Every 1 Hour</option>
                                <option value="every6hours" <?= selected(get_option($this->interval_option_name), 'every6hours') ?>>Every 6 Hours</option>
                                <option value="every12hours" <?= selected(get_option($this->interval_option_name), 'every12hours') ?>>Every 12 Hours</option>
                            </select>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Post Author</th>
                        <td>
                            <?php wp_dropdown_users(['name' => $this->user_option_name, 'selected' => get_option($this->user_option_name)]); ?>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Max Posts Per Feed</th>
                        <td>
                            <input type="number" name="<?= $this->max_posts_option ?>" value="<?= esc_attr(get_option($this->max_posts_option, 10)) ?>" min="1" />
                            <p class="description">Maximum number of posts to import from each feed (default: 10).</p>
                        </td>
                    </tr>
                </table>
                <h2>RSS/XML Feeds</h2>
                <table class="form-table">
                    <tbody id="rss-feeds-list">
                    <?php foreach ($feeds as $index => $feed): ?>
                        <tr>
                            <td><input type="text" name="<?= $this->option_name ?>[<?= $index ?>][url]" value="<?= esc_attr($feed['url']) ?>" placeholder="Feed URL" size="40" /></td>
                            <td><?php wp_dropdown_categories(['name' => $this->option_name . "[$index][cat]", 'selected' => $feed['cat'], 'hide_empty' => 0]); ?></td>
                            <td><input type="text" name="<?= $this->option_name ?>[<?= $index ?>][lang]" value="<?= esc_attr($feed['lang']) ?>" placeholder="Language" /></td>
                            <td><button class="button remove-feed">Remove</button></td>
                            <td><button class="button import-feed" data-url="<?= esc_url($feed['url']) ?>" data-cat="<?= esc_attr($feed['cat']) ?>" data-lang="<?= esc_attr($feed['lang']) ?>">Import</button></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
                <p><button class="button add-feed">Add Feed</button></p>
                <?php submit_button(); ?>
            </form>
        </div>
        <script>
        document.querySelector('.add-feed').addEventListener('click', function(e) {
            e.preventDefault();
            const list = document.getElementById('rss-feeds-list');
            const count = list.children.length;
            const tr = document.createElement('tr');
            tr.innerHTML = `
                <td><input type="text" name="<?= $this->option_name ?>[${count}][url]" value="" placeholder="Feed URL" size="40" /></td>
                <td><?php wp_dropdown_categories(['name' => $this->option_name . "[__INDEX__][cat]", 'hide_empty' => 0]); ?></td>
                <td><input type="text" name="<?= $this->option_name ?>[${count}][lang]" value="" placeholder="Language" /></td>
                <td><button class="button remove-feed">Remove</button></td>
                <td><button class="button import-feed" data-url="" data-cat="1" data-lang="">Import</button></td>
            `.replace(/__INDEX__/g, count);
            list.appendChild(tr);
        });

        document.addEventListener('click', function(e) {
            if (e.target.classList.contains('remove-feed')) {
                e.preventDefault();
                e.target.closest('tr').remove();
            }

            if (e.target.classList.contains('import-feed')) {
                e.preventDefault();
                const tr = e.target.closest('tr');
                const url = tr.querySelector('input[name*="[url]"]').value;
                const cat = tr.querySelector('select').value;
                const lang = tr.querySelector('input[name*="[lang]"]').value;
                fetch(ajaxurl + `?action=manual_import_feed&url=${encodeURIComponent(url)}&cat=${cat}&lang=${encodeURIComponent(lang)}`)
                    .then(res => res.json())
                    .then(data => alert(data.data || data.message));
            }
        });
        </script>
        <?php
    }

    public static function deactivate() {
        wp_clear_scheduled_hook('auto_rss_import_hook');
    }
}

function autorss_importer_init() {
    if (is_admin() || defined('DOING_CRON') || defined('DOING_AJAX')) {
        new AutoRSSImporter();
    }
}
add_action('plugins_loaded', 'autorss_importer_init');

register_activation_hook(__FILE__, function () {
    if (get_option('auto_rss_import_feeds') === false) {
        add_option('auto_rss_import_feeds', []);
    }
    if (get_option('auto_rss_import_interval') === false) {
        add_option('auto_rss_import_interval', 'hourly');
    }
    if (get_option('auto_rss_import_user') === false) {
        $admin_user = get_users(['role' => 'administrator', 'number' => 1]);
        $admin_id = !empty($admin_user) ? $admin_user[0]->ID : 1;
        add_option('auto_rss_import_user', $admin_id);
    }
    if (get_option('auto_rss_import_fetch_full') === false) {
        add_option('auto_rss_import_fetch_full', true);
    }
    if (get_option('auto_rss_import_max_items') === false) {
        add_option('auto_rss_import_max_items', 10);
    }
});

register_deactivation_hook(__FILE__, ['AutoRSSImporter', 'deactivate']);
